//
//  GameBoardViewController.m
//  RoadTripBingo
//
//  Created by Juan Navas Martin on 2/15/14.
//  Copyright (c) 2014 Juan Navas. All rights reserved.
//

#import "GameBoardViewController.h"

static const int MSIZE = 5;

@interface GameBoardViewController () {

    NSDate *baseDate;
    NSTimer *gameTime;
    //to save Bingo signs in matrix positions
    int bingoPositions[MSIZE];
    BOOL prebingo, bingoConfirmed, confirmBingoAlertShown;
}

@end


@implementation GameBoardViewController



- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
    self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) {
        // Custom initialization
    }
    return self;
}

- (void)viewDidLoad
{
    [super viewDidLoad];
    prebingo = NO, bingoConfirmed = NO, confirmBingoAlertShown = NO;
    
    NSLog(@"GameBoardViewController: ViewDidLoad, current Game: %@", self.currentGame);
    
    self.playerNameLabel.text = self.currentGame.playerName;
    baseDate = self.currentGame.startDate;
    
    [self updateTimerLabel];
    
}

-(void)viewWillAppear:(BOOL)animated
{
    if (self.currentGame.hasBeenUpdated) {
        RTBGameDAO *gameDAO = [[RTBGameDAO alloc]init];
        [gameDAO saveCurrentGame:self.currentGame];
        gameDAO = nil;
        self.currentGame.hasBeenUpdated = NO;
        prebingo = NO, bingoConfirmed = NO, confirmBingoAlertShown = NO;
    }
    
    //NSLog(@"1.- selected Item: %@", self.selectedTS);
    if (self.selectedTS.hasBeenUpdated) {
        GameDetailsDAO *dao = [[GameDetailsDAO alloc]init];
        [dao updateGameDetails:self.selectedTS];
        dao = Nil;
        
        [self.collectionView reloadData];
        self.selectedTS.hasBeenUpdated = NO;
    }
    
    //if a possible BINGO hasn't been achieved yet, check again.
    if (!prebingo) {
        prebingo = [self calculateBingo];
    }
    
    //If Bingo has been achieved, we save the end date (provisionally) and ask the user to confirm SIGNS
    if (prebingo) {
        if (!self.currentGame.endDate) {
            self.currentGame.endDate = [NSDate date];
            
            
            RTBGameDAO *dao = [[RTBGameDAO alloc]init];
            [dao saveCurrentGame:self.currentGame];
            dao = Nil;
        }
        if (!confirmBingoAlertShown) {
            UIAlertView *confirmBingoAlert = [[UIAlertView alloc]
                                              initWithTitle:@"Possible Bingo!" message:@"please confirm the signs to Win!" delegate:nil cancelButtonTitle:@"OK" otherButtonTitles:nil];
            [confirmBingoAlert show];
            confirmBingoAlertShown = YES;
        }
        
        
        //If Bingo hasn't been confirmed, we check if all the signs have been checked.
        if ((bingoConfirmed=[self bingoHasBeenConfirmed])) {
            self.currentGame.gotBingo = YES;
            RTBGameDAO *dao = [[RTBGameDAO alloc]init];
            [dao saveCurrentGame:self.currentGame];
            dao = Nil;
            
            UIAlertView *bingoAlert = [[UIAlertView alloc]
                                       initWithTitle:@"Congratulations" message:@"You've completed BINGO!!" delegate:nil cancelButtonTitle:@"OK" otherButtonTitles:nil];
            [bingoAlert show];
        }
        
    }
    else
    {
        [self pauseOrResumeGameTimer];
    }
}

-(void)viewWillDisappear:(BOOL)animated {
    if (!prebingo) {
        //if (!self.currentGame.endDate) {
        [self pauseOrResumeGameTimer];
    }
}

- (NSInteger)collectionView:(UICollectionView *)collectionView numberOfItemsInSection:(NSInteger)section {
    return [self.tSigns count];
}

// The cell that is returned must be retrieved from a call to -dequeueReusableCellWithReuseIdentifier:forIndexPath:
- (UICollectionViewCell *)collectionView:(UICollectionView *)collectionView cellForItemAtIndexPath:(NSIndexPath *)indexPath {
    
    //NSLog(@"indexpath: %@", indexPath);
    
    TrafficSignCell *tsCell = [collectionView dequeueReusableCellWithReuseIdentifier:@"tsCell" forIndexPath:indexPath];
    
    RTBSign *gts = [self.tSigns objectAtIndex:indexPath.row];
    
    tsCell.tsCellImage.image = [UIImage imageNamed:gts.defaultImageUrl];
    
    tsCell.tsCellConfirmationImage.hidden = YES;
    if (gts.timestamp) {
        if (prebingo) {
            if ([self isWinnerCell:gts.position]) {
                tsCell.tsCellImage.alpha = 1;
                tsCell.tsCellConfirmationImage.hidden = NO;
                if (gts.hasBeenConfirmed){
                    tsCell.tsCellConfirmationImage.image = [UIImage imageNamed:@"confirmed"];
                }
                else {
                    tsCell.tsCellConfirmationImage.image = [UIImage imageNamed:@"unconfirmed"];
                }
            }
            else
            {
                tsCell.tsCellImage.alpha = 0.3;
            }
        }
        else
        {
            tsCell.tsCellImage.alpha = 1;
        }
        
    }
    else
    {
        //if the bingo has been achieved, we fade out a little more irrelevant images
        prebingo ? (tsCell.tsCellImage.alpha = 0.3) : (tsCell.tsCellImage.alpha = 0.5);
    }
    return tsCell;
}

-(BOOL)shouldPerformSegueWithIdentifier:(NSString *)identifier sender:(id)sender {
    NSIndexPath *selectedIndexPath = [[self.collectionView indexPathsForSelectedItems] objectAtIndex:0];
    RTBSign *s =[self.tSigns objectAtIndex:selectedIndexPath.row];
    if (prebingo && ![self isWinnerCell:s.position]) {
        return NO;
    }
    else {
        return YES;
    }
}
- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

#pragma mark - protocol methods
- (void) passTSToGameBoardViewController:(TSDetailsViewController *)controller didFinishEditingTS:(RTBSign *)selectedItem inGame:(RTBGame *)game
{
    self.selectedTS = selectedItem;
    self.currentGame = game;
}

#pragma mark - Navigation
-(void)prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender {

    if([segue.identifier isEqualToString:@"showSignDetail"]){
        TSDetailsViewController *tsdvc = [segue destinationViewController];
        
        NSIndexPath *selectedIndexPath = [[self.collectionView indexPathsForSelectedItems] objectAtIndex:0];
        //NSLog(@"Index: %li", (long)selectedIndexPath.row);
        tsdvc.detailedTS = [self.tSigns objectAtIndex:selectedIndexPath.row];
        tsdvc.detailedTS.hasBeenUpdated = NO;
        tsdvc.currentGame = self.currentGame;
        tsdvc.prebingo = prebingo;
        
        tsdvc.delegate = self;
    }
}
- (IBAction)backButton:(id)sender {
    //[[self navigationController] popViewControllerAnimated:YES];
    [[self navigationController] popToRootViewControllerAnimated:YES];
}


#pragma mark - User methods

-(void)pauseOrResumeGameTimer {
    
    if (!gameTime) {
        gameTime = [NSTimer scheduledTimerWithTimeInterval:.1
                                                    target:self
                                                  selector:@selector(updateTimerLabel)
                                                  userInfo:Nil
                                                   repeats:YES];
    }
    else {
        [gameTime invalidate];
        gameTime = nil;
    }
}

//Method to display a time counter label in the game
-(void)updateTimerLabel {
    NSTimeInterval interval;
    if (self.currentGame.endDate) {
        interval = [self.currentGame.endDate timeIntervalSinceDate:baseDate];
    }else {
        interval = [baseDate timeIntervalSinceNow];
    }
    //double intpart;
    //double fractional = modf(interval, &intpart);
    //NSUInteger hundredth = ABS((int)(fractional*100));
    unsigned int seconds = ABS((int)interval);
    unsigned int minutes = seconds/60;
    unsigned int hours = minutes/60;
    //    self.gameTimerLabel.text = [NSString stringWithFormat:@"%02d:%02d:%02d:%02d", hours, minutes%60, seconds%60, hundredth];
    self.gameTimerLabel.text = [NSString stringWithFormat:@"%02u:%02u:%02u", hours, minutes%60, seconds%60];
}

-(BOOL) calculateBingo {
    
    //counters arrays for checked cells on each row, each column, main diagonal and reverse diagonal
    int row [MSIZE], column [MSIZE], diag1 = 0, diag2 = 0;
    
    //counters for each row or column in the counters
    int r=0, c=0, totalHits=0;
    
    //to Know if we have got Bingo or Not
    BOOL hasBingo = FALSE;
    
    
    //initialization of arrays
    for (int i = 0; i < MSIZE; i++) {
        row[i] = 0;
        column[i] = 0;
        bingoPositions[i] = 0;
    }
    
    //object retrieved at each cell of the array of signs
    RTBSign *gts = [[RTBSign alloc]init];
    
    //Loop through the array of signs
    for (int i=0; i<25; i++){
        
        //if the index in the main array is multiple of the matrix dimension we should increment rows
        //and reset the column, except for i = 0
        if (i!=0 && i % (MSIZE) == 0) {
            r++;
            c=0;
        }
        
        gts = [self.tSigns objectAtIndex:i];
    
        if (gts.timestamp) {
            
            //if the sign has been checked, increment the score in row and column
            row [r]++;
            column [c]++;
            totalHits++;
            
            // if row and column are the same, increment the score in main diagonal
            if (r==c) {
                diag1++;
            }
            
            //The row and column for each cell in the reverse diagonal will sum the matrix dimension
            if (r + c == MSIZE-1) {
                diag2++;
            }
        }
        
        
        //each loop should increment the column to be checked
        c++;
    }
    
    //Check rows and columns to see if the user has got a Bingo!
    for (int i=0; i<MSIZE; i++) {
        //NSLog(@"\nRow [%i] = %i\nColumn [%i] = %i\n", i, row[i], i, column[i]);
        if (row[i] == MSIZE) {
            hasBingo = true;
            // "i" is the Row in the bingo combination. To know positions in Collection View we use this formula:
            //
            // indexPath = Row * MSIZE + Column
            //
            // And we store this positions in an Array: BingoPositions
            for (int colIndex=0; colIndex < MSIZE ; colIndex++) {
                bingoPositions[colIndex] = i * MSIZE + colIndex;
                NSLog(@"\nbingoPositions[%d] = %d", colIndex, bingoPositions[i]);
            }
        }
        if (column[i] == MSIZE) {
            hasBingo = true;
            // "i" is the Column in the bingo combination. To know positions in Collection View we use this formula:
            //
            // indexPath = Row * MSIZE + Column
            //
            // And we store this positions in an Array: BingoPositions
            for (int rowIndex=0; rowIndex < MSIZE; rowIndex++) {
                bingoPositions [rowIndex] = rowIndex * MSIZE + i;
                NSLog(@"\nbingoPositions[%d] = %d", rowIndex, bingoPositions[i]);
            }
        }
    }
    
    if (diag1 == MSIZE) {
        hasBingo = true;
        // Diag1 is the Bingo combination. To know positions in Collection View we use this formula:
        //
        // indexPath = Row * MSIZE + Column
        //
        // where Row = Column = i
        //
        // And we store this positions in an Array: BingoPositions
        for (int i=0; i < MSIZE; i++) {
            bingoPositions [i] = i * MSIZE + i;
            NSLog(@"\nbingoPositions[%d] = %d", i, bingoPositions[i]);
        }
    }
    if (diag2 == MSIZE) {
        hasBingo = true;
        // Diag2 is the Bingo combination. To know positions in Collection View we use this formula:
        // indexPath = Row * MSIZE + Column
        // where Row = i (increments) and Column = MSIZE - 1 (decrements)
        // And we store this positions in an Array: BingoPositions
        int colIndex = MSIZE - 1;
        for (int i=0; i < MSIZE; i++) {
            bingoPositions [i] = i * MSIZE + colIndex;
            NSLog(@"\nbingoPositions[%d] = %d", i, bingoPositions[i]);
            colIndex--;
        }
    }
    
    for (int i=0; i<5; i++) {
        NSLog(@"\nFINAL: bingoPositions[%d] = %d", i, bingoPositions[i]);
    }
    
    //NSLog(@"\nmain Diag:%i\nReverseDiag=%i", diag1, diag2);
    
    int maxRow=0, maxColumn=0;
    
    maxRow = maxValueOfArray(row, MSIZE);
    maxColumn = maxValueOfArray(column, MSIZE);
    //NSLog(@"maxRow = %d\nmaxColumn=%d\ndiag1=%i\ndiag2=%i\n", maxRow, maxColumn, diag1, diag2);
    
    self.lineLabel.text = [NSString stringWithFormat:@"Line:\t%d", MAX(maxColumn, MAX(maxRow, MAX(diag1, diag2)))];
    self.totalLabel.text = [NSString stringWithFormat:@"Total:\t%d", totalHits];
    
    
    return hasBingo;

}

int maxValueOfArray(int myArray[], size_t size) {
    /* enforce the contract */
    //assert(myArray && size);
    
    //By definition, the size of an array cannot be negative. The appropriate variable for array sizes in C is size_t
    size_t i;
    int maxValue = myArray[0];
    
    NSLog(@"maxValue = %d\nsize=%zu",maxValue, size);
    
    for (i = 1; i < size; ++i) {
        if ( myArray[i] > maxValue ) {
            maxValue = myArray[i];
        }
    }
    return maxValue;
}


- (BOOL) isWinnerCell:(int)position {
    for (int i = 0; i < MSIZE; i++) {
        if (position == bingoPositions[i]) {
            return YES;
        }
    }
    return NO;
}

- (BOOL) bingoHasBeenConfirmed {
    BOOL confirmed = YES;
    int position;
    for (int i=0; i < MSIZE; i++) {
        position = bingoPositions[i];
        if (![[self.tSigns objectAtIndex:position] hasBeenConfirmed]) {
            confirmed = NO;
        }
    }
    return confirmed;
}

@end